/*
 * The /dev/random DLKM for HP-UX 11.00
 *
 * $Id: random.c,v 1.3 2003/12/15 07:45:12 josv Exp $
 *
 * (c) Copyright 2003 Jos Visser <josv@osp.nl> and others.
 * The core entropy routines are (c) Copyright 1994-1999 Theodore Ts'o.
 *
 * This software is licensed under the GNU General Public License (GPL). 
 * See the file LICENSE for more information.
 */

/*
 * Include HP-UX kernel headers
 */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/errno.h>
#include <sys/malloc.h>
#include <mod_conf.h>
#include <sys/moddefs.h>
#include <sys/io.h>
#include <sys/wsio.h>

/*
 * The version number
 */
#define RANDOM_VERSION "0.0.3"

/*
 * Forward declarations of the DLKM entry points. These are
 * necessary because they are referenced in important structures that
 * are declared in the top of the source file.
 */
int random_install(void);
static int random_load(void *drv_infop);
static int random_unload(void *drv_infop);
static int random_open(dev_t dev, int flags, intptr_t dummy, int mode);
static int random_close(dev_t dev, int flags, int mode);
static int random_read(dev_t dev, struct uio *uio);
static int random_write(dev_t dev, struct uio *uio);
static int random_init(void);
static void random_linked_init(void);

/*
 * These are the tunable parameters of this module. See also
 * kmtune(1m).
 */
extern int random_io_threshold;
extern int random_debug;

/*
 * DLKM wrapper tables
 */
extern struct mod_operations gio_mod_ops;
static drv_info_t random_drv_info;
extern struct mod_conf_data random_conf_data;

/*
 * Module type specific data
 */
static struct mod_type_data random_drv_link = {
	"random - /dev/random for HP-UX 11.00", (void *) NULL
};

static struct modlink random_mod_link[] = {
	{&gio_mod_ops, (void *) &random_drv_link},	/* WSIO */
	{NULL, (void *) NULL}
};

struct modwrapper random_wrapper = {
	MODREV,
	random_load,
	random_unload,
	(void (*)()) NULL,
	(void *) &random_conf_data,
	random_mod_link
};

/*
 * Driver information blocks for HP-UX.
 * The major numbers are set to -1, which means dynamic major number
 * allocation.
 */
static drv_info_t random_drv_info = {

	"random",		/* type */
	"pseudo",		/* class */
	DRV_CHAR | DRV_PSEUDO | DRV_SAVE_CONF | DRV_MP_SAFE,	/* flags */
	-1,			/* b_major */
	-1,			/* c_major */
	NULL,			/* cdio */
	NULL,			/* gio_private */
	NULL,			/* cdio_private */
};

/*
 * This structure contains pointers to the
 * operations supported by this driver. Currently only
 * open, close, read and write are supported.
 */
static drv_ops_t random_drv_ops = {
	random_open,		/* d_open */
	random_close,		/* d_close */
	NULL,			/* d_strategy */
	NULL,			/* d_dump */
	NULL,			/* d_psize */
	NULL,			/* d_mount */
	random_read,		/* d_read */
	random_write,		/* d_write */
	NULL,			/* d_ioctl */
	NULL,			/* d_select */
	NULL,			/* d_option1 */
	NULL,			/* reserved1 */
	NULL,			/* reserved2 */
	NULL,			/* reserved3 */
	NULL,			/* reserved4 */
	0			/* d_flags */
};

/*
 * The following two structures contain information for
 * the HP-UX I/O layers
 */
static wsio_drv_data_t random_wsio_data = {
	"pseudo_random",	/* drv_path */
	T_DEVICE,		/* drv_type */
	DRV_CONVERGED,		/* drv_flags */
	NULL,			/* dvr_minor_build - field not used */
	NULL,			/* drv_minor_decode - field not used */
};

static wsio_drv_info_t random_wsio_info = {
	&random_drv_info,
	&random_drv_ops,
	&random_wsio_data,
};

/*
 * Some constants
 */
#define MINOR_URANDOM		1
#define MINOR_RANDOM		2
#define MINOR_PANIC		3
#define MAX_TRANSFER_SIZE	8192

#define RANDOM_SPINLOCK_NAME	"random DLKM spinlock"
#define RANDOM_SPINLOCK_ORDER	PRINTF_LOCK_ORDER

/*
 * Please note that I am wondering about the correct level
 * for the random_spinlock. PRINTF_LOCK_ORDER might be
 * too high, but it seems to be working...
 */

/*
 * Local module variables
 */
static int (*random_saved_init) ();
static d_strategy_t old_sdisk_strategy;
static int io_counter;
static caddr_t random_wait_channel=(caddr_t)&random_wait_channel;
static lock_t *random_spinlock;

/*
 * My own homebrew naive memset
 */
static void *memset(void *dest, int c, size_t n)
{
	char *s=(char *)dest;

	while (n) {
		*s=c;
		s++;
		n--;
	}

	return dest;
}

/*
 * My own homebrew naive memcpy
 */
static void *memcpy(void *dest, const void *src, size_t n)
{
	char *s, *t;

	s=(char *)dest;
	t=(char *)src;

	while (n) {
		*s=*t;
		s++;
		t++;
		n--;
	}

	return dest;
}

/*
 * Include the random pool stuff derived from Linux
 */
#include "lnx.c"

/*
 * Take entropy from the specified kernel buffer
 */
static void take_entropy(struct buf *bp)
{
	__u32 data[64];
	__u32 digest[HASH_BUFFER_SIZE+HASH_EXTRA_SIZE];
	int i;

	/*
	 * Lock the random spinlock. This ensures that we are running
	 * this code "alone", even in an SMP machine.
	 */
	spinlock(random_spinlock);

	/*
	 * Increment the I/O counter.
	 */
	io_counter++;

	/*
	 * Only take an entropy sample after "random_io_threshold"
	 * I/O's. 
	 */
	if (io_counter>=random_io_threshold) {
		/*
		 * Reset the I/O counter
		 */
		io_counter=0;

		/*
		 * Update the entropy pool with the time interval
		 * of these I/O's and mix in the current clock
		 * value together with the entropy as well.
		 */
		add_strategy_randomness(ticks_since_boot);

		/*
		 * Copy the first 64 bytes of the data block read/written
		 * to a local buffer
		 */
		if (privlbcopy(bp->b_spaddr,bp->b_un.b_addr,ldsid(data),(caddr_t)data,64)==0) {
			/*
			 * Hash the first 16 bytes of the data block and add them
			 * to the entropy store
			 */
			HASH_TRANSFORM(digest,data);

			for (i=0; i<HASH_BUFFER_SIZE; i+=2)
				batch_entropy_store(digest[i]&255,digest[i+1]&255,16);
		} else
			printf("random> privlbcopy failed!\n");

		/*
		 * Since changes to the entropy store are actually not executed
		 * inline but batched, process the outstanding batch now...
		 */
		batch_entropy_process(random_state);
	}

	/*
	 * Do not forget to unlock!
	 */
	spinunlock(random_spinlock);
}

/*
 * This is the new strategy routine that all SCSI disk strategy
 * gets revectored to...
 */
static int random_strategy(struct buf *bp) 
{
	/*
	 * Check whether this operation is for a SCSI write
	 * operation. Please note that you can not test
	 * for B_WRITE!!!! (the proof of this is left to the
	 * reader as an exercise!
	 * If this is a write op, take entropy from the kernel
	 * buffer.
	 */
	if (!(bp->b_flags&B_READ))
		take_entropy(bp);

	/*
	 * Now execute the original SCSI strategy routine.
	 */
	return (*old_sdisk_strategy)(bp);
}

/*
 * Replace the strategy routine of the sdisk driver with our
 * own strategy routine
 */
static int random_revector() {
	extern struct bdevsw bdevsw[];
	struct bdevsw *p=&bdevsw[31];
	drv_info_t *info=p->d_drv_info;

	/*
	 * The major number of the sdisk driver is fixed at 31.
	 * However, an extra check provides safety in kernel
	 * configurations that I can not fathom right now...
	 */
	if (strcmp(info->name,"sdisk")!=0) {
		printf("random> Driver 31 is not sdisk!\n");
		return 0;
	}

	/*
	 * Save the current strategy routine and replace it with
	 * a pointer to our own routine
	 */
	old_sdisk_strategy=p->d_strategy;
	p->d_strategy=random_strategy;

	return 1;
}

/*
 * Swap the saved strategy routine back into the system
 */
static void random_unrevector() {
	extern struct bdevsw bdevsw[];
	struct bdevsw *p=&bdevsw[31];

	p->d_strategy=old_sdisk_strategy;
}
	
/* 
 * This method is called after DLKM load in order to initialise
 * the DLKM. Most of the work is done by the random_init() 
 * method, which makes it possible to link this driver statically
 * into the kernel as well...
 */
static int random_load(void *arg)
{
	if (random_debug)
		printf("random> Loading...\n");

	/*
	 * Use drv_info passed to us instead of static version
	 */
	random_wsio_info.drv_info = (drv_info_t *) arg;

	/*
	 * Register this driver with the HP-UX WSIO layer
	 */
	if (!wsio_install_driver(&random_wsio_info)) {
		printf("random> wsio_install_driver failed!!\n");
		return (ENXIO);
	}

	/* 
	 * Perform driver-specific initialization, but do not
	 * callnext function in the dev_init list. 
	 */
	if (!random_init())
		return ENXIO;

	return 0;
}

/*
 * This function is called before DLKM unload. It should
 * free resources and deregister everything. 
 *
 * This function is only called when the administrator
 * attempts to unload the module and there are no open 
 * devices using the module. If there is some reason that
 * the module should not be unloaded, check it now and 
 * return non-zero. 
 */ 
static int random_unload(void *drv_infop)
{
	/*
	 * Unregister with WSIO 
	 */
	if (wsio_uninstall_driver(&random_wsio_info)) {
		/*
		 * Uninstall failed! Return to a loaded,
		 * functional state. 
		 */
		printf("random> wsio_uninstall_driver failed!!\n");
		return (ENXIO);
	}

	/* 
	 * Cancel pending timeouts, free allocated memory
	 * and resources, etc. 
	 */
	dealloc_spinlock(random_spinlock);
	random_unrevector();

	printf("random> Unloaded...\n");

	return (0);
}

/*
 * This function is called if module is statically 
 * linked. It links the random_init function into the chain
 * of driver initialisation routines.
 */ 
int random_install(void)
{
	extern int (*dev_init) (void);

	/*
	 * Link my init function into chain 
	 */ 
	random_saved_init = dev_init;
	dev_init = (int (*)()) &random_linked_init;

	/*
	 * Register driver with WSIO
	 */
	return (wsio_install_driver(&random_wsio_info));
}

/*
 * Device initialization Link 
 * Called only for statically linked drivers to link 
 * initroutine into list. 
 */
static void random_linked_init(void)
{
	/* 
	 * Perform driver-specific initialization
	 */
	random_init();

	/* Call next init function in chain */
	(void) (*random_saved_init) ();
}

/*
 * Device initialization 
 * This is common code for both statically linked and 
 * dynamically loaded modules. 
 */
static int random_init(void)
{				
	/*
	 * Allocate a new spinlock
	 */
	random_spinlock=alloc_spinlock(RANDOM_SPINLOCK_ORDER,RANDOM_SPINLOCK_NAME);

	printf("random> version=%s; wait channel=%08x; spinlock=%08x; io_treshold=%d\n",
			RANDOM_VERSION,random_wait_channel,random_spinlock,random_io_threshold);
	
	/*
	 * Call the entropy store initialisation routine
	 */
	if (rand_initialize()) {
		/*
		 * Revector ourselves into the SCSI disk I/O routine
		 */
		if (!random_revector()) {
			printf("random> Load process failed!\n");
			return 0;
		}
	} else {
		printf("random> rand_initialize failed\n");
		return 0;
	}

	return 1;
}

/*
 * OPEN 
 * This does nothing but check the minor number and give a debug message.
 */ 
static int random_open(dev_t dev, int flag, intptr_t dummy, int mode)
{
	int m=minor(dev);

	/*
	 * Minor number is outside the supported values
	 */
	if (m<MINOR_URANDOM || m>MINOR_PANIC)
		return ENODEV;

	if (random_debug)
		printf("random> open()\n");

	return (0);
}

/*
 * CLOSE 
 * This does nothing but give a debug message.
 */ 
static int random_close(dev_t dev, int flag, int mode)
{
	if (random_debug)
		printf("random> close()\n");

	return (0);
}

/*
 * Read of the urandom device. The urandom device returns
 * the requested number of bytes without blocking for new
 * entropy to be collected and stored in the pool. Therefore
 * it never blocks, but random bytes from the entropy store
 * are recycled as necessary, potentially leading data that is
 * less random... The maximum number of bytes returned is
 * MAX_TRANSFER_SIZE (8KB).
 */
static int dev_urandom_read(struct uio *uio)
{
	long size=uio->uio_resid;
	char *buf;
	int err;
	
	/*
	 * What an easy request this was!
	 */
	if (size==0)
		return;

	/*
	 * Is a negative size read a write?
	 */
	if (size<0 || size>MAX_TRANSFER_SIZE)
		return EINVAL;

	/*
	 * Allocate a buffer for storing the entropy
	 */
	buf=KMALLOC(size);

	if (!buf)
		return EMSGSIZE;
	
	/*
	 * Lock the random spinlock to protect the entropy store
	 */
	spinlock(random_spinlock);

	/*
	 * Return "size" bytes of entropy from the entropy store. Please
	 * note that if less than "size" bytes are available the bytes
	 * in the pool are recycled as needed leading to potentially
	 * less random data...
	 */
	extract_entropy(sec_random_state, buf, size, EXTRACT_ENTROPY_SECONDARY);

	/*
	 * Unlock the spinlock
	 */
	spinunlock(random_spinlock);

	/*
	 * Move the buffer with the data to user space
	 */
	err=uiomove((caddr_t)buf,size,UIO_READ,uio);

	/*
	 * Free the kernel buffer
	 */
	KFREE(buf);

	return err;
}

/*
 * Read of the random device. This device returns the maximum
 * entropy data. If not enough bytes are available in the 
 * entropy store, this function blocks until enough data is
 * available...
 */
static int dev_random_read(struct uio *uio)
{
	long size=uio->uio_resid;
	char *buf;
	int n;
	int err;
	
	/*
	 * Check for illegal size requests
	 */
	if (size==0)
		return 0;

	if (size<0)
		return EINVAL;

	/*
	 * Allocate a buffer to transfer chunks of data from the
	 * entropy pool to the user
	 */
	buf=KMALLOC(SEC_XFER_SIZE);

	if (!buf)
		return EMSGSIZE;

	/*
	 * Transfer *all* the bytes please. 
	 */
	while (size>0) {
		/*
		 * size bytes are requested
		 */
		n=size;

		/*
		 * However, limit n to the chunk size
		 */
		if (n>SEC_XFER_SIZE)
			n=SEC_XFER_SIZE;

		/*
		 * Lock the spinlock (protect the entropy pool)
		 */
		spinlock(random_spinlock);

		/*
		 * And limit "n" to the number of bytes now
		 * available in the entropy pool
		 */
		if (n>random_state->entropy_count/8)
			n=random_state->entropy_count/8;

		/*
		 * If there is no data in the entropy pool
		 */
		if (n==0) {
			/*
			 * Free the lock on the entropy pool
			 */
			spinunlock(random_spinlock);

			/*
			 * And sleep... We will be woken up by
			 * the entropy pool management routines
			 * when they put some data in the pool
			 */
			sleep(random_wait_channel,PRIBIO);

			/*
			 * Yawn.... we woke up... There are
			 * bytes in the pool (hopefully)
			 */
			continue;
		}

		/*
		 * There was data in the entropy pool (n bytes). Take them out
		 * to the local buffer
		 */
		n=extract_entropy(sec_random_state,buf,n,EXTRACT_ENTROPY_SECONDARY);

		/*
		 * Free the lock protecting the entropy store
		 */
		spinunlock(random_spinlock);

		/*
		 * And move the local buffer to user space. This will automatically
		 * advance the pointers in the uio structure
		 */
		err=uiomove((caddr_t)buf,n,UIO_READ,uio);

		/*
		 * If there was an error, free the local buffer and
		 * bail out
		 */
		if (err!=0) {
			KFREE(buf);
			return err;
		}

		/*
		 * Did n bytes, subtract from size and continue
		 */
		size-=n;
	}

	/*
	 * Free the local buffer
	 */
	KFREE(buf);

	return 0;
}

/*
 * READ 
 * Look at the minor number of the device being read and call
 * the correct routine for read of random or urandom device...
 */ 
static int random_read(dev_t dev, struct uio *uio)
{
	int m=minor(dev);

	if (random_debug)
		printf("random>  read(minor=%d,size=%d)\n",m,uio->uio_resid);

	switch (m) {
		case MINOR_URANDOM:
			return dev_urandom_read(uio);

		case MINOR_RANDOM:
			return dev_random_read(uio);

		default:
			return ENODEV;
	}

	return 0;
}

/*
 * /dev/panic, who does not want that!!! :-) 
 * Writing a string to this file causes a system panic with
 * the specified string as reason. What does this function
 * do here? Well, I wanted it once but did not have it at
 * that time (HP-UX 9 days), and I could throw it in for free
 * with the rest of this stuff... :-)
 */ 
static int panic_write(struct uio *uio)
{
	long size=uio->uio_resid;
	char *buf;
	int err;
	
	/*
	 * Unlikely, but worth checking...
	 */
	if (size>MAX_TRANSFER_SIZE)
		return EINVAL;
		
	/*
	 * Allocate one byte more for the \0 termination
	 */
	buf=KMALLOC(size+1);
	memset(buf,0,size+1);

	/*
	 * Move the data of write() to the kernel buffer
	 */
	err=uiomove((caddr_t)buf,size,UIO_WRITE,uio);

	/*
	 * If we got the data, call the kernel panic routine.
	 */
	if (err==0) {
		panic(buf);
		printf("random> Panic failed. Please reboot the Matrix...\n");
	}

	/*
	 * Convergent programming...
	 */
	KFREE(buf);

	return err;
}

/*
 * WRITE 
 * Look at the minor number of the device being read and call
 * the correct routine. Currently only "/dev/panic" is supported
 * for write....
 */ 
static int random_write(dev_t dev, struct uio *uio)
{
	int m=minor(dev);

	if (random_debug)
		printf("random>  write(minor=%d,size=%d)\n",m,uio->uio_resid);

	switch (m) {
		case MINOR_PANIC:
			return panic_write(uio);

		default:
			return ENODEV;
	}

	return 0;
}
